<?php

/**
 * @version    5.3.0
 * @package    Com_Gavisitsignin
 * @author     Glenn Arkell <glenn@glennarkell.com.au>
 * @copyright  2020 Glenn Arkell
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace GlennArkell\Component\Gavisitsignin\Site\Model;

// No direct access.
defined('_JEXEC') or die;

use \Joomla\CMS\Factory;
use \Joomla\CMS\Language\Text;
use \Joomla\CMS\MVC\Model\ListModel;
use \Joomla\CMS\Component\ComponentHelper;
use \GlennArkell\Component\Gavisitsignin\Administrator\Helper\GavisitsigninHelper;

/**
 * Methods supporting a list of records.
 */
class MembersModel extends ListModel
{
	/**
	 * Constructor.
	 * @param   array  $config  An optional associative array of configuration settings.
	 * @see        Controller
	 * @since      1.6
	 */
	public function __construct($config = array())
	{
		if (empty($config['filter_fields']))
		{
			$config['filter_fields'] = array(
				'id', 'a.id',
				'block', 'a.block',
				'name', 'a.name',
			);
		}

		parent::__construct($config);
	}

	/**
	 * Method to auto-populate the model state.
	 * Note. Calling getState in this method will result in recursion.
	 * @param   string  $ordering   Elements order
	 * @param   string  $direction  Order direction
	 * @return void
	 * @throws Exception
	 * @since    1.6
	 */
	protected function populateState($ordering = null, $direction = null)
	{
		$app  = Factory::getApplication();
		$list = $app->getUserState($this->context . '.list');
		$printOutput     = $app->input->getInt('p', 0);
        $this->setState('printOutput', $printOutput);

		$ordering  = isset($list['filter_order'])     ? $list['filter_order']     : null;
		$direction = isset($list['filter_order_Dir']) ? $list['filter_order_Dir'] : null;

		if(empty($ordering)) {
			$ordering = $app->getUserStateFromRequest($this->context . '.filter_order', 'filter_order', $app->get('filter_order'));
			if (!in_array($ordering, $this->filter_fields)) {
				$ordering = "a.name";
			}
			$this->setState('list.ordering', $ordering);
		}
		if(empty($direction)) {
			$direction = $app->getUserStateFromRequest($this->context . '.filter_order_Dir', 'filter_order_Dir', $app->get('filter_order_Dir'));
			if (!in_array(strtoupper($direction ?? ''), array('ASC', 'DESC', ''))) {
				$direction = "ASC";
			}
			$this->setState('list.direction', $direction);
		}

		$list['limit']     = $app->getUserStateFromRequest($this->context . '.list.limit', 'limit', $app->get('list_limit'), 'uint');
		//$list['limit']     = 0;
		$list['start']     = $app->input->getInt('start', 0);
		$list['ordering']  = $ordering;
		$list['direction'] = $direction;
		
		$app->setUserState($this->context . '.list', $list);
		if ($printOutput) {
            $app->input->set('list', 0);
        }
		//$app->input->set('list', null);   // if you use this, list always defaults to Global

        // List state information.
        parent::populateState($ordering, $direction);

        $context = $this->getUserStateFromRequest($this->context.'.filter.search', 'filter_search');
        $this->setState('filter.search', $context);
        $status = $this->getUserStateFromRequest($this->context.'.filter.state', 'filter_state');
        $this->setState('filter.state', $status);
        $ugroup = $this->getUserStateFromRequest($this->context.'.filter.ugroup', 'filter_ugroup');
        $this->setState('filter.ugroup', $ugroup);

	}

	/**
	 * Build an SQL query to load the list data.
	 * @return   DatabaseQuery
	 * @since    1.6
	 */
	protected function getListQuery()
	{
		$user       = GavisitsigninHelper::getSpecificUser();
		$canCheckin = $user->authorise('core.manage', 'com_gavisitsignin');
		$params = ComponentHelper::getParams('com_gavisitsignin');
		$group_id = $params->get('group_id', 1);    // default to public to get all

        $db    = $this->getDbo();
        $query = $db->getQuery(true);

		// Select the required fields from the table.
        $query->select( ' a.id, a.id AS user_id, a.name, a.email, a.username ' );
        $query->select(" SUBSTRING_INDEX(SUBSTRING_INDEX(a.name, ' ', 1), ' ', -1) AS first_name " );
        $query->select(" SUBSTRING_INDEX(SUBSTRING_INDEX(a.name, ' ', 4), ' ', -1) AS last_name " );
        $query->from('#__users AS a');

		// Join over the usergroup field 'user_id'
		if ($group_id > 1) {
			$query->select('g.group_id AS group_id_access');
			$query->join('LEFT', '#__user_usergroup_map AS g ON g.user_id = a.id AND g.group_id = '.(int) $group_id);
			$query->where(' g.group_id IS NOT NULL ');
		}

        $status = $this->getState('filter.state', 0);
		if ($status) {
			$query->where('a.block = '.(int) $status);
		} else {
			$query->where('a.block = 0');
		}

        // If not manager, show only individuals record
        if (!$canCheckin) {
			$query->where('a.id = '.(int) $user->id);
        }

        // Filter by search in title
        $search = $this->getState('filter.search');
        if (!empty($search)) {
            if (stripos($search, 'id:') === 0) {
                $query->where('a.id = ' . (int) substr($search, 3));
            } else {
                $search = $db->Quote('%' . $db->escape($search, true) . '%');
				$query->where('( a.name LIKE ' . $search . ' )');
            }
        }

        // Filter by search in title
        $ugroup = $this->getState('filter.ugroup', 0);
		if (!empty($ugroup) && $ugroup) {
			$query->join('LEFT', '#__user_usergroup_map AS ug ON ug.user_id = a.id AND ug.group_id = '.(int) $ugroup);
			$query->where(' ug.group_id IS NOT NULL ');
		}

        // Add the list ordering clause.
        $query->order(" SUBSTRING_INDEX(SUBSTRING_INDEX(a.name, ' ', 4), ' ', -1) ASC " );

        return $query;
	}

	/**
	 * Method to get an array of data items
	 * @return  mixed An array of data on success, false on failure.
	 */
	public function getItems()
	{
		$items = parent::getItems();
		

		return $items;
	}

	/**
	 * Overrides the default function to check Date fields format, identified by
	 * "_dateformat" suffix, and erases the field if it's not correct.
	 * @return void
	 */
	protected function loadFormData()
	{
		$app              = Factory::getApplication();
		$filters          = $app->getUserState($this->context . '.filter', array());
		$error_dateformat = false;

		foreach ($filters as $key => $value)
		{
			if (strpos($key, '_dateformat') && !empty($value) && $this->isValidDate($value) == null)
			{
				$filters[$key]    = '';
				$error_dateformat = true;
			}
		}

		if ($error_dateformat)
		{
			$app->enqueueMessage(Text::_("COM_GAVISITSIGNIN_SEARCH_FILTER_DATE_FORMAT"), "warning");
			$app->setUserState($this->context . '.filter', $filters);
		}

		return parent::loadFormData();
	}

	/**
	 * Checks if a given date is valid and in a specified format (YYYY-MM-DD)
	 * @param   string  $date  Date to be checked
	 * @return bool
	 */
	private function isValidDate($date)
	{
		$date = str_replace('/', '-', $date);
		return (date_create($date)) ? Factory::getDate($date)->format("Y-m-d") : null;
	}

    /**
     * Method to sign out all remaining signed in attendances
     * @return  boolean
     */
    public function signOutAllAttendees()
    {
		$user = GavisitsigninHelper::getSpecificUser();

        if ($user->authorise('core.manage', 'com_gavisitsignin') !== true) {
            throw new Exception(Text::_('JERROR_ALERTNOAUTHOR'), 403);
        }

	    try {
	        $return = GavisitsigninHelper::signInAllRemaining($user->id);
	    } catch (RuntimeException $e) {
	        Factory::getApplication()->enqueueMessage($e->getMessage(), 'danger');
	        return false;
	    }

        return $return;
        
    }

}
