<?php

/**
 * @version    5.3.0
 * @package    Com_Gavisitsignin
 * @author     Glenn Arkell <glenn@glennarkell.com.au>
 * @copyright  2020 Glenn Arkell
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace GlennArkell\Component\Gavisitsignin\Site\Model;

// No direct access.
defined('_JEXEC') or die;

use \Joomla\CMS\Factory;
use \Joomla\CMS\Language\Text;
use \Joomla\CMS\MVC\Model\ListModel;
use \Joomla\CMS\Component\ComponentHelper;
use \GlennArkell\Component\Gavisitsignin\Administrator\Helper\GavisitsigninHelper;

/**
 * Methods supporting a list of records.
 */
class AttendancesModel extends ListModel
{
	/**
	 * Constructor.
	 * @param   array  $config  An optional associative array of configuration settings.
	 * @see        JController
	 * @since      1.6
	 */
	public function __construct($config = array())
	{
		if (empty($config['filter_fields']))
		{
			$config['filter_fields'] = array(
				'id', 'a.id',
				'ordering', 'a.ordering',
				'state', 'a.state',
				'created_by', 'a.created_by',
				'modified_by', 'a.modified_by',
				'created_date', 'a.created_date',
				'modified_date', 'a.modified_date',
				'user_id', 'a.user_id', 'user_id_name', 'u.name',
				'clockin_date', 'a.clockin_date', 'dateFr', 'dateTo',
				'clockout_date', 'a.clockout_date',
				'visit_who', 'a.visit_who',
				'name', 'a.name', 'attend_name', 'SUBSTRING_INDEX(SUBSTRING_INDEX(attend_name, " ", 4), " ", -1)',
				'phone', 'a.phone',
				'address', 'a.address',
				'suburb', 'a.suburb',
				'postcode', 'a.postcode',
				'comment', 'a.comment',
			);
		}

		parent::__construct($config);
	}

	/**
	 * Method to auto-populate the model state.
	 * Note. Calling getState in this method will result in recursion.
	 * @param   string  $ordering   Elements order
	 * @param   string  $direction  Order direction
	 * @return void
	 * @throws Exception
	 * @since    1.6
	 */
	protected function populateState($ordering = null, $direction = null)
	{
		$app  = Factory::getApplication();
		$adminList = $app->getMenu()->getActive()->getParams()->get('admin_list', 0);
		$app->setUserState('menu.parameter.admin', $adminList);
        $list = $app->getUserState($this->context . '.list');
        //$app->setUserState('menu.parameter.admin');

		$ordering  = isset($list['filter_order'])     ? $list['filter_order']     : null;
		$direction = isset($list['filter_order_Dir']) ? $list['filter_order_Dir'] : null;

		if(empty($ordering)) {
			$ordering = $app->getUserStateFromRequest($this->context . '.filter_order', 'filter_order', $app->get('filter_order'));
			if (!in_array($ordering, $this->filter_fields)) {
				$ordering = 'SUBSTRING_INDEX(SUBSTRING_INDEX(attend_name, " ", 4), " ", -1)';
			}
			$this->setState('list.ordering', $ordering);
		}
		if(empty($direction)) {
			$direction = $app->getUserStateFromRequest($this->context . '.filter_order_Dir', 'filter_order_Dir', $app->get('filter_order_Dir'));
			if (!in_array(strtoupper($direction ?? ''), array('ASC', 'DESC', ''))) {
				$direction = 'asc';
			}
			$this->setState('list.direction', $direction);
		}

		if (!$adminList) {
            $list['limit']     = 0;
        } else {
            $list['limit']     = $app->getUserStateFromRequest($this->context . '.list.limit', 'limit', $app->get('list_limit'), 'uint');
        }
		$list['start']     = $app->input->getInt('start', 0);
		$list['ordering']  = $ordering;
		$list['direction'] = $direction;
		
		$app->setUserState($this->context . '.list', $list);
		//$app->input->set('list', null);   // if you use this, list always defaults to Global

        // List state information.
        parent::populateState($ordering, $direction);

        $context = $this->getUserStateFromRequest($this->context.'.filter.search', 'filter_search');
        $this->setState('filter.search', $context);
        $status = $this->getUserStateFromRequest($this->context.'.filter.state', 'filter_state');
        $this->setState('filter.state', $status);
        $ugroup = $this->getUserStateFromRequest($this->context.'.filter.ugroup', 'filter_ugroup');
        $this->setState('filter.ugroup', $ugroup);
        
        $dateFr = $this->getUserStateFromRequest($this->context.'.filter.dateFr', 'filter_dateFr');
        $this->setState('filter.dateFr', $dateFr);
        $dateTo = $this->getUserStateFromRequest($this->context.'.filter.dateTo', 'filter_dateTo');
        $this->setState('filter.dateTo', $dateTo);
        $mias = $this->getUserStateFromRequest($this->context.'.filter.mias', 'filter_mias');
        $this->setState('filter.mias', $mias);
	}

	/**
	 * Build an SQL query to load the list data.
	 * @return   JDatabaseQuery
	 * @since    1.6
	 */
	protected function getListQuery()
	{
        // get params to see if all records to show
        //$adminList = Factory::getApplication()->getMenu()->getActive()->getParams()->get('admin_list', 0);
        $adminList = Factory::getApplication()->getUserState('menu.parameter.admin');
        $user = Factory::getApplication()->getIdentity();
        $today = date_format(Factory::getDate(),'Y-m-d');

        $params = ComponentHelper::getParams('com_gavisitsignin');
        $dstStart = $params->get('dstStart', '10-05');
        $dstEnd = $params->get('dstEnd', '04-05');

        $tz = date_format(Factory::getDate(),'m-d') >= $dstStart && date_format(Factory::getDate(),'m-d') <= $dstEnd ? '+11:00' : '+10:00';

        // Create a new query object.
        $db    = $this->getDbo();
        $query = $db->getQuery(true);

		// Select the required fields from the table.
        $query->select( $this->getState( 'list.select', 'DISTINCT a.*' ) );
        $query->select(" IF (a.user_id=0,CONCAT(a.name,' visiting ',a.visit_who), u.name) AS attend_name ");
        $query->select(" SUBSTRING_INDEX(SUBSTRING_INDEX(u.name, ' ', 1), ' ', -1) AS first_name " );
        $query->select(" SUBSTRING_INDEX(SUBSTRING_INDEX(u.name, ' ', 4), ' ', -1) AS last_name " );
        $query->select(" date_format(CONVERT_TZ(clockin_date, '+00:00', ". $db->Quote($tz). "), '%Y-%m-%d') AS clockinDate ");
        $query->select(" date_format(CONVERT_TZ(clockout_date, '+00:00', ". $db->Quote($tz). "), '%Y-%m-%d') AS clockoutDate ");
		$query->from('#__gavisitsignin_attendances AS a');

		// Join over the users for the checked out user.
		$query->select('uc.name AS uEditor');
		$query->join('LEFT', '#__users AS uc ON uc.id=a.checked_out');

		// Join over the user field 'created_by'
		$query->select('created_by.name AS created_by');
		$query->join('LEFT', '#__users AS created_by ON created_by.id = a.created_by');

		// Join over the user field 'modified_by'
		$query->select('modified_by.name AS modified_by');
		$query->join('LEFT', '#__users AS modified_by ON modified_by.id = a.modified_by');

		// Join over the user field 'user_id'
		$query->select('u.name AS user_id_name, u.registerDate');
		$query->join('LEFT', '#__users AS u ON u.id = a.user_id');

        $status = $this->getState('filter.state');
		if (isset($status) && ($status == 0 || $status == 1 || $status == 2 || $status == -2 )) {
			$query->where('a.state = '.(int) $status);
		} elseif ($status == '*')  {
			// don't filter
		} else {
			$query->where('a.state IN (0,1)');
		}

		if (!$adminList) {
			$query->where('a.user_id = '.(int) $user->id);
		}

        // Filter by search in title
        $search = $this->getState('filter.search');
        if (!empty($search)) {
            if (stripos($search, 'id:') === 0) {
                $query->where('a.id = ' . (int) substr($search, 3));
            } else {
                $search = $db->Quote('%' . $db->escape($search, true) . '%');
				$query->where('( a.name LIKE ' . $search . ' OR u.name LIKE ' . $search . ' OR substr(a.clockin_date,1,10) LIKE ' . $search . ' OR substr(a.clockout_date,1,10) LIKE ' . $search . ')');
            }
		}

        // Filter by usergroup
        $ugroup = $this->getState('filter.ugroup', 0);
		if (!empty($ugroup) && $ugroup) {
			$query->join('LEFT', '#__user_usergroup_map AS ug ON ug.user_id = a.id AND ug.group_id = '.(int) $ugroup);
			$query->where(' ug.group_id IS NOT NULL ');
		}

        // Filter by date range
        $dateFr = $this->getState('filter.dateFr');
		if ($dateFr) {
			$query->where(" date_format(CONVERT_TZ(a.clockin_date, '+00:00', ". $db->Quote($tz). "), '%Y-%m-%d') >= ".$db->quote($dateFr));
		}
        $dateTo = $this->getState('filter.dateTo');
		if ($dateTo) {
			$query->where(" date_format(CONVERT_TZ(a.clockin_date, '+00:00', ". $db->Quote($tz). "), '%Y-%m-%d') <= ".$db->quote($dateTo));
		}

        // Add the list ordering clause.
        $defOrder = 'SUBSTRING_INDEX(SUBSTRING_INDEX(attend_name, " ", 4), " ", -1)';
        $orderCol  = $this->state->get('list.ordering', $defOrder);
		$orderDirn = $this->state->get('list.direction', "asc");

		if ($orderCol && $orderDirn) {
			$query->order( $orderCol . ' ' . $orderDirn . ', a.clockin_date ASC' );
		}

        return $query;
	}

	/**
	 * Method to get an array of data items
	 * @return  mixed An array of data on success, false on failure.
	 */
	public function getItems()
	{
		$items = parent::getItems();

		return $items;
	}

	/**
	 * Overrides the default function to check Date fields format, identified by
	 * "_dateformat" suffix, and erases the field if it's not correct.
	 * @return void
	 */
	protected function loadFormData()
	{
		$app              = Factory::getApplication();
		$filters          = $app->getUserState($this->context . '.filter', array());
		$error_dateformat = false;

		foreach ($filters as $key => $value)
		{
			if (strpos($key, '_dateformat') && !empty($value) && $this->isValidDate($value) == null)
			{
				$filters[$key]    = '';
				$error_dateformat = true;
			}
		}

		if ($error_dateformat)
		{
			$app->enqueueMessage(Text::_("COM_GAVISITSIGNIN_SEARCH_FILTER_DATE_FORMAT"), "warning");
			$app->setUserState($this->context . '.filter', $filters);
		}

		return parent::loadFormData();
	}

	/**
	 * Checks if a given date is valid and in a specified format (YYYY-MM-DD)
	 * @param   string  $date  Date to be checked
	 * @return bool
	 */
	private function isValidDate($date)
	{
		$date = str_replace('/', '-', $date);
		return (date_create($date)) ? Factory::getDate($date)->format("Y-m-d") : null;
	}

	/**
	 * Export attendances data for analysis
	 * @return bool
	 */
	public function generateExport()
	{
		$app      = Factory::getApplication();
		$filters  = $app->getUserState($this->context . '.filter', array());
        $header = array('user_id', 'clockinDate', 'clockoutDate', 'UserName', 'registerDate', 'lastLoginDate');
        $params = ComponentHelper::getParams('com_gavisitsignin');
        $fileName = $params->get('fileName', 'images/members/export/signindata.csv');

		$atts = GavisitsigninHelper::getAttendancesExport($filters['dateFr'], $filters['dateTo']);
        
        $fp = \fopen($fileName, 'w');
        \fputcsv($fp, $header);
        foreach ($atts as $fields) {
            \fputcsv($fp, $fields);
        }
        \fclose($fp);

        return true;
	}

	/**
	 * Method to get any user records not signed in for a while
	 * @return  mixed An array of data on success, false on failure.
	 */
	public function showMIA()
	{
		$app      = Factory::getApplication();
		$filters  = $app->getUserState($this->context . '.filter', array());
        $params = ComponentHelper::getParams('com_gavisitsignin');
        $mias = !empty($filters['mias']) ? $filters['mias'] : 60;

		$items = GavisitsigninHelper::getMIArecords($mias);
        $app->setUserState('com_gavisitsignin.mia.data', $items);
        
		return $items;
	}

}
