<?php

/**
 * @version    5.3.0
 * @package    Com_Gavisitsignin
 * @author     Glenn Arkell <glenn@glennarkell.com.au>
 * @copyright  2020 Glenn Arkell
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace GlennArkell\Component\Gavisitsignin\Site\Model;

// No direct access.
defined('_JEXEC') or die;

use \Joomla\CMS\Factory;
use \Joomla\Utilities\ArrayHelper;
use \Joomla\CMS\Language\Text;
use \Joomla\CMS\Table\Table;
use \Joomla\CMS\User\UserHelper;
use \Joomla\CMS\MVC\Model\ItemModel;
use \Joomla\Filesystem\File;
use \Joomla\Filesystem\Folder;
use \Joomla\Filesystem\Path;
use \Joomla\CMS\Component\ComponentHelper;
use \GlennArkell\Component\Gavisitsignin\Administrator\Helper\GavisitsigninHelper;

/**
 * Single Record model.
 */
class AttendanceModel extends ItemModel
{
	/**
	 * A loaded item
	 * @var    \stdClass
	 * @since  1.6
	 */
	protected $_item = null;

	/**
	 * Method to auto-populate the model state.
	 * Note. Calling getState in this method will result in recursion.
	 * @return void
	 * @since    1.6
     * @throws Exception
	 */
	protected function populateState()
	{
		$app  = Factory::getApplication('com_gavisitsignin');
		$user  = $app->getIdentity();

		// Check published state
		if ((!$user->authorise('core.edit.state', 'com_gavisitsignin')) && (!$user->authorise('core.edit', 'com_gavisitsignin')))
		{
			$this->setState('filter.published', 1);
			$this->setState('filter.archived', 2);
		}

		// Load state from the request userState on edit or from the passed variable on default
		if ($app->input->get('layout') == 'edit') {
			$id = $app->getUserState('com_gavisitsignin.edit.attendance.id');
		} elseif ($app->input->get('layout') == 'modalMIA') {
		} else {
			$id = $app->input->get('id');
			$app->setUserState('com_gavisitsignin.edit.attendance.id', $id);
		}

		$this->setState('attendance.id', $id);

		// Load the parameters.
		$params       = $app->getParams();
		$params_array = $params->toArray();

		if (isset($params_array['item_id']))
		{
			$this->setState('attendance.id', $params_array['item_id']);
		}

		$this->setState('params', $params);
	}

	/**
	 * Method to get an object.
	 * @param   integer $id The id of the object to get.
	 * @return  mixed    Object on success, false on failure.
     * @throws Exception
	 */
	public function getItem($id = null)
	{
		$app  = Factory::getApplication('com_gavisitsignin');

        if ($this->_item === null) {
            $this->_item = false;

            if (empty($id)) {
                $id = $this->getState('attendance.id');
            }

            // Get a level row instance.
            $table = $this->getTable();

            // Attempt to load the row.
            if ($table->load($id)) {

                // Check published state.
                if ($published = $this->getState('filter.published')) {
                    if (isset($table->state) && $table->state != $published) {
                        throw new Exception(Text::_('COM_GAVISITSIGNIN_ITEM_NOT_LOADED'), 403);
                    }
                }

                // Convert the JTable to a clean JObject.
                $properties  = $table->getProperties(1);
                $this->_item = ArrayHelper::toObject($properties, 'JObject');

            }
        }
        
		if (isset($this->_item->created_by)) {
			$this->_item->created_by_name = GavisitsigninHelper::getSpecificUser($this->_item->created_by)->name;
		}

		if (isset($this->_item->modified_by)) {
			$this->_item->modified_by_name = GavisitsigninHelper::getSpecificUser($this->_item->modified_by)->name;
		}

		if (isset($this->_item->user_id)) {
			$this->_item->user_id_name = GavisitsigninHelper::getSpecificUser($this->_item->user_id)->name;
		}

        return $this->_item;
    }

	/**
	 * Get an instance of Table class
	 * @param   string $type   Name of the JTable class to get an instance of.
	 * @param   string $prefix Prefix for the table class name. Optional.
	 * @param   array  $config Array of configuration values for the JTable object. Optional.
	 * @return  Table|bool Table if success, false on failure.
	 */
	public function getTable($type = 'Attendance', $prefix = 'Administrator', $config = array())
	{
		return parent::getTable($type, $prefix, $config);
	}

	/**
	 * Get the id of an item by alias
	 * @param   string $alias Item alias
	 * @return  mixed
	 */
	public function getItemIdByAlias($alias)
	{
            $table      = $this->getTable();
            $properties = $table->getProperties();
            $result     = null;

            if (key_exists('alias', $properties))
            {
                $table->load(array('alias' => $alias));
                $result = $table->id;
            }
            
                return $result;
            
	}

	/**
	 * Method to check in an item.
	 * @param   integer $id The id of the row to check out.
	 * @return  boolean True on success, false on failure.
	 * @since    1.6
	 */
	public function checkin($id = null)
	{
		// Get the id.
		$id = (!empty($id)) ? $id : (int) $this->getState('attendance.id');
                
		if ($id)
		{
			// Initialise the table
			$table = $this->getTable();

			// Attempt to check the row in.
			if (method_exists($table, 'checkin'))
			{
				if (!$table->checkin($id))
				{
					return false;
				}
			}
		}

		return true;
                
	}

	/**
	 * Method to check out an item for editing.
	 * @param   integer $id The id of the row to check out.
	 * @return  boolean True on success, false on failure.
	 * @since    1.6
	 */
	public function checkout($id = null)
	{
		// Get the user id.
		$id = (!empty($id)) ? $id : (int) $this->getState('attendance.id');

		if ($id) {
			// Initialise the table
			$table = $this->getTable();

			// Get the current user object.
			$user  = Factory::getApplication()->getIdentity();

			// Attempt to check the row out.
			if (method_exists($table, 'checkout')) {
				if (!$table->checkout($user->id, $id)) {
					return false;
				}
			}
		}

		return true;
                
	}

	/**
	 * Publish the element
	 * @param   int $id    Item id
	 * @param   int $state Publish state
	 * @return  boolean
	 */
	public function publish($id, $state)
	{
		$table = $this->getTable();
                
		$table->load($id);
		$table->state = $state;

		return $table->store();
                
	}

	/**
	 * Method to delete an item
	 * @param   int $id Element id
	 * @return  bool
	 */
	public function delete($id)
	{
		$table = $this->getTable();

		$table->load($id);
		$table->state = -2;

		return $table->store();
                
	}

    /**
     * Method to automatically load or update an attendance record.
     * @param string encrypted user code or zero if not using encryption
     * @param object User record object of the member signing in or out
     * @return bool
     */
    public function signInOut($qrEncrypt = 0,$user = 0)
    {
		$app = Factory::getApplication('com_gavisitsignin');
		$creator  = $app->getIdentity();
        $today = Factory::getDate()->toSql();
        $todaySignIn = date_format(Factory::getDate('now', 'Australia/Melbourne'), 'Y-m-d');
		$params = ComponentHelper::getParams('com_gavisitsignin');
		$defTime = $params->get('def_so_time', '16:00:00');

        if ($qrEncrypt) {
			$comment = 'Attendance Registered using encryption '.$qrEncrypt;
		} else {
			$comment = 'Attendance Registered';
		}

        // Check the user can edit this item
        $authorised = $user->authorise('core.create', 'com_gavisitsignin');
        if ($authorised !== true) {
            $app->enqueueMessage(Text::_('JERROR_ALERTNOAUTHOR'), 'danger');
            return false;
        }

		// Check if already signed in - an attendance record exists for today
		$attRec = GavisitsigninHelper::checkSignInDate($user->id, $todaySignIn);
		if ($attRec) {

			if (isset($attRec->clockout_date) && !empty($attRec->clockout_date)) {

				//signing in for second time for the day
				$table = $this->getTable();
		        $data = array();
				$data['state'] = 1;
				$data['created_by'] = $creator->id;
				$data['created_date'] = $today;
				$data['user_id'] = $user->id;
				$data['clockin_date'] = $today;
				$data['comment'] = $comment;

		        if ($table->save($data) === true) {
		            return $user->name.' - Signed In';
		        } else {
		            return false;
		        }
	        } else {
                
                // signout for todays record
				$table = $this->getTable();
				$table->load($attRec->id);
	            $table->clockout_date = $today;
	            $table->modified_by = $creator->id;
	            $table->comment = $table->comment."\r\n".$comment;
		        if ($table->store($attRec->id) === true) {
					return $user->name.' - Signed Out';
		        } else {
		            return false;
				}
	        }
		} else {
            // if no previous signin today check for non signout in past
            $oldRec = GavisitsigninHelper::getLastSignInDate($user->id, $todaySignIn);
            
            if (isset($oldRec->clockin_date) && empty($oldRec->clockout_date)) {
                // signout old record before signing in new
                // first setup signout datetime
                $localDT = new \DateTimeZone('Australia/Sydney');
                $utcDT = new \DateTimeZone('UTC');

                $signIn = new \DateTime($oldRec->clockin_date, $utcDT);
                $signIn->setTimezone($localDT);
                $signinDateTime = $signIn->format('Y-m-d H:i:s');

                $signOut = new \DateTime($signIn->format('Y-m-d').' '.$defTime, $localDT);
                $signOut->setTimezone($utcDT);
                $signoutDateTime = $signOut->format('Y-m-d H:i:s');

				$sotable = $this->getTable();
				$sotable->load($oldRec->id);
                // needs to sign out old one
	            $sotable->clockout_date = $signoutDateTime;
	            $sotable->modified_by = $creator->id;
	            $sotable->comment = $sotable->comment."\r\n".'Signed out old record due to new sign in.';
		        if ($sotable->store($attRec->id) === true) {
					$app->enqueueMessage($user->name.' - Signed Out old record due to new sign in.', 'notice');
				}
            }
			$table = $this->getTable();
	        $data = array();
			$data['id'] = 0;
			$data['state'] = 1;
			$data['created_by'] = $creator->id;
			$data['created_date'] = $today;
			$data['user_id'] = $user->id;
			$data['clockin_date'] = $today;
			$data['comment'] = $comment;

	        if ($table->save($data) === true) {
	            return $user->name.' - Signed In';
	        } else {
	            return false;
	        }
        }

    }
}
